//Author: Jake Adriaens
//Updated: 2-18-06

#include <iostream>
#include <fstream>
#include "map.h"
#include "map_init.h"

#ifdef GUI
#include <GL/glut.h>
#endif

#include <sys/time.h>
#include <new>
using namespace std;

//create map object
map current_map;

void usage() {
	cout << "Usage: breach <-b> <-v> <-f> <-a> <-g> [grid_size] -map [map file]\n";
	cout << "  -b: specifies black and white mode\n";
	cout << "  -v: specifies verbose mode, i.e. turns on gui\n";
	cout << "  -f: specified fov only (and map bounding box)\n";
	cout << "  -g: specifies find path with grid instead of voronoi\n";
	cout << "  -a: specifies to animate the simulation\n";
}


int animate_on;

#ifdef GUI
void displayCB(void);

int last_time;

void animate(int n) {
	struct timeval tv;
	int current_time;
	int diff_time;
	gettimeofday(&tv, NULL);
	current_time = tv.tv_usec;
	if(current_time >= last_time) diff_time = current_time - last_time;
	else {
		//cout << "here\n";
		diff_time = (1000000-last_time)+current_time;
	}
	//if(diff_time > 100000) {
	//	cout << diff_time << endl << current_time << endl << last_time << endl;
	//}

	current_map.move(((double)diff_time)/1000000.0);
	//gettimeofday(&tv, NULL);
	//last_time = tv.tv_usec;
	last_time = current_time;
	displayCB();
	//cout << diff_time << endl;
	//cout << last_time << endl;
	//cout << diff_time << endl;
}

void displayCB(void)		/* function called whenever redisplay needed */
{
	glClear(GL_COLOR_BUFFER_BIT);		/* clear the display */
	glColor3f(1.0, 1.0, 1.0);		/* set current color to white */
  
	current_map.draw();  /* draw the map */
 
	glFlush();				/* Complete any pending operations */
	
	//glutTimerFunc(unsigned int msecs,
	//							void (*func)(int value), value);
	if(animate_on) {
		struct timeval tv;
		int current_time;
		int diff_time;
		gettimeofday(&tv, NULL);
		current_time = tv.tv_usec;
		if(current_time >= last_time) diff_time = current_time - last_time;
		else diff_time = (1000000-last_time)+current_time;
		diff_time = diff_time / 1000;
		diff_time = 50 - diff_time;
		if(diff_time <= 0) diff_time = 1;	
		glutTimerFunc(diff_time, &animate, 0);
	}
}

void keyCB(unsigned char key, int x, int y)	/* called on key press */
{
	if( key == 'q' ) exit(0);
	else if(key=='a') {
		animate_on = !animate_on;
		timeval tv;
		gettimeofday(&tv, NULL);
		last_time = tv.tv_usec;
		animate(0);
	} else if( key == 'n' ) {
		current_map.nodes_on = !current_map.nodes_on;
		displayCB();
	} else if( key == 'v' ) {
		current_map.voronoi_on = !current_map.voronoi_on;
		displayCB();
	} else if( key == 'f' ) {
		current_map.fov_on = !current_map.fov_on;
		displayCB();
	//} else if( key == 'i' ) {
	//	m->int_on = !m->int_on;
	//	displayCB();
	} else if( key == 'p' ) {
		current_map.path_on = !current_map.path_on;
		displayCB();
	/*} else if( key == 'N' ) {
		if(current_map.node_color >= 1.0) current_map.node_color = 0.0;
		else current_map.node_color = current_map.node_color + 0.1;
		displayCB();
	} else if( key == 'V' ) {
		if(m->voronoi_color >= 1.0) m->voronoi_color = 0.0;
		else m->voronoi_color = m->voronoi_color + 0.1;
		displayCB();
	} else if( key == 'F' ) {
		if(m->fov_color >= 1.0) m->fov_color = 0.0;
		else m->fov_color = m->fov_color + 0.1;
		displayCB();
	} else if( key == 'I' ) {
		if(m->intersect_color >= 1.0) m->intersect_color = 0.0;
		else m->intersect_color = m->intersect_color + 0.1;
		displayCB();
	} else if( key == 'P' ) {
		if(m->path_color >= 1.0) m->path_color = 0.0;
		else m->path_color = m->path_color + 0.1;
		displayCB();*/
	} else if( key == 'b' ) {
		current_map.node_size++;
		displayCB();
	} else if( key == 'c' ) {
		current_map.voronoi_size++;
		displayCB();
	} else if( key == 'd' ) {
		current_map.fov_size++;
		displayCB();
	//} else if( key == 'u' ) {
	//	m->intersect_size++;
	//	displayCB();
	} else if( key == 'o' ) {
		current_map.path_size++;
		displayCB();
	} else if( key == 'B' ) {
		current_map.node_size--;
		if(current_map.node_size < 0) current_map.node_size = 0;
		displayCB();
	} else if( key == 'C' ) {
		current_map.voronoi_size--;
		if(current_map.voronoi_size < 0) current_map.voronoi_size = 0;
		displayCB();
	} else if( key == 'D' ) {
		current_map.fov_size--;
		if(current_map.fov_size < 0) current_map.fov_size = 0;
		displayCB();
	//} else if( key == 'U' ) {
	//	m->intersect_size--;
	//	if(m->intersect_size < 0) m->intersect_size = 0;
	//	displayCB();
	} else if( key == 'O' ) {
		current_map.path_size--;
		if(current_map.path_size < 0) current_map.path_size = 0;
		displayCB();
	} else if(key=='w') {
		current_map.weight_color_on = !current_map.weight_color_on;
		displayCB();
	} else if(key=='r') {
		current_map.rotation_on = !current_map.rotation_on;
		animate(0);
	} else if(key=='m') {
		current_map.movement_on = !current_map.movement_on;
		animate(0);
	}
		
}
#endif

int main(int argc, char *argv[]) {
  
  /********************************
	* Here we are parsing arguments *
	********************************/
  //check number of arguments
	if(argc < 3 || argc > 8) {
		cout << "Invalid number of arguments.\n";
		usage();
		return -1;
	}

	ifstream map_file;
	int i;
	int verbose = 0;
	int grid = 0;
	int bw = 0;
	int only_fov = 0;
	animate_on = 0;
  //parse arguments
	for(i=1; i<argc; i++) {
		if(strcmp(argv[i], "-map") == 0) {
			map_file.open(argv[i+1]);
			if(!map_file) {
				cout << "Unable to open: " << argv[i+1] << endl;
				usage();
				return -1;
			}
			i++;
		} else if(strcmp(argv[i], "-v") == 0) {
			if(verbose == 1) {
				cout << "Duplicated argument.\n";
				usage();
				return -1;
			}
			verbose = 1;
		} else if(strcmp(argv[i], "-g") == 0) {
			if(grid) {
				cout << "Duplicated argument.\n";
				usage();
				return -1;
			}
			grid = atoi(argv[i+1]);
			if(grid <= 0) {
				cout << "Grid size must be > 0.\n";
				usage();
				return -1;
			}
			i++;
		} else if(strcmp(argv[i], "-f") == 0) {
			if(only_fov) {
				cout << "Duplicated argument.\n";
				usage();
				return -1;
			}
			only_fov = 1;
		} else if(strcmp(argv[i], "-b") == 0) {
			if(bw) {
				cout << "Duplicated argument.\n";
				usage();
				return -1;
			}
			bw = 1;
		} else if(strcmp(argv[i], "-a") == 0) {
			if(animate_on) {
				cout << "Duplicated arument.\n";
				usage();
				return -1;
			}
			animate_on = 1;
		}else {
			cout << "Invalid argument: " << argv[i] << endl;
			usage();
			return -1;
		}
	}
  
  
  /****************************
	* Here we run the algorithm *
	****************************/
	
	//initialize map with nodes and fov's
	if(!create_map_from_file(&map_file, &current_map)) return -1;
	current_map.only_fov = only_fov;
	current_map.grid = grid;
	current_map.add_overlap_edges();
	current_map.intersect_edges();
	current_map.remove_outside_edges();
	current_map.update_weights();
	double maximal_breach = current_map.find_maximal_breach();
	
	//cout << "1\n";
	//current_map.move(200);
	//cout << "2\n";
	//current_map.move(200);
	//cout << "3\n";
	//current_map.move(200);
	//cout << "4\n";
	//current_map.move(200);
	//cout << "5\n";

  /******************************
	* Here we display the results *
	******************************/
	cout << maximal_breach << endl;
	
#ifdef GUI
	if(verbose) {
		//m->bw = bw;
		int win;
		glutInit(&argc, argv);		/* initialize GLUT system */
		glutInitDisplayMode(GLUT_RGB);
		glutInitWindowSize((int)current_map.get_width()+210,(int)current_map.get_height()+14);		/* width=1024pixels height=768pixels */
		win = glutCreateWindow("Breach");	/* create window */
			
		if(!bw) glClearColor(0.0,0.0,0.0,0.0);	/* set background to black */
		else glClearColor(1.0,1.0,1.0,1.0); //set background white
		gluOrtho2D(-105,(int)current_map.get_width()+105,-8,(int)current_map.get_height()+6); /* how object is mapped to window */
		glutDisplayFunc(displayCB);		/* set window's display callback */
		glutKeyboardFunc(keyCB);		/* set window's key callback */
		
		glutMainLoop();			/* start processing events... */
	}
#endif
	
	return 0;
}
