#ifndef LINUX_COMPILE
#include "stdafx.h"
#endif

#include "map.h"
#include "edge.h"
#include "node.h"
#include "point.h"
#include "math_functions.h"
#include "globals.h"

#include <math.h>

#ifdef GUI
#include <GL/glut.h>
#endif

#include <iostream>
#include <new>

#include "graph.h"

#include <stdlib.h>

using namespace std;

double intersect_x[512];
double intersect_y[512];
int intersect_count;

map::map() {
	srand48(time(NULL));
	
	intersect_count = 0;
	
	width = 0;
	height = 0;
	node_head = NULL;
	node_tail = NULL;
	nodes = 0;
	edge_head = NULL;
	edge_tail = NULL;
	edges = 0;
	
	//initial display
	fov_on = true;
	nodes_on = true;
	voronoi_on = true;
	path_on = true;
	weight_color_on = true;
	rotation_on = true;
	movement_on = true;
	node_size = 4.0;
	voronoi_size = 2.0;
	fov_size = 2.0;
	path_size = 5.0;
	
	vertices = 0;
	
	GV = NULL;
}

void map::remove_outside_edges() {
	edge1 = edge_head;
	while(edge1) {
		if(edge1->type != START) {
		if(edge1->get_min_x() < -DELTA ||
				 edge1->get_min_y() < -DELTA ||
				 edge1->get_max_x() > width+DELTA ||
				 edge1->get_max_y() > height+DELTA) {
			edge1 = remove_edge(edge1);
		} else {
			edge1 = edge1->next; 
		}
		} else {
			edge1 = edge1->next;
		}
	}
}

edge *map::remove_edge(edge *e) {
	point *temp;
	temp = e->get_start();
	if(temp) delete temp;
	temp = e->get_stop();
	if(temp) delete temp;
	edge *t;
	t = e->next;
	if(e != edge_head) e->prev->next = e->next;
	else edge_head = e->next;
	if(e != edge_tail) e->next->prev = e->prev;
	else edge_tail = e->prev;
	delete e;
	edges--;
	return t;
}

void map::set_width(double new_width) {
	width = new_width;
}

void map::set_height(double new_height) {
	height = new_height;
}

double map::get_width() {
	return width;
}

double map::get_height() {
	return height;
}

void map::add_node(double x1, double y1, double angle, double sweep, double range) {
	//we add the three edges and the node
	double x2, y2, x3, y3;
	x2 = end_x(x1, range, angle);
	y2 = end_y(y1, range, angle);
	x3 = end_x(x1, range, angle+sweep);
	y3 = end_y(y1, range, angle+sweep);
	
	//cout << x1 << "," << y1 << " " << x2 << "," << y2 << " " << x3 << "," << y3 << endl;
	
	if(!node_head) {
		node_head = new (std::nothrow) node(x1, y1, x2, y2, x3, y3);
		if(!node_head) {
			cout << "Error, out of memory." << endl;
			exit(-1);
		}
		node_tail = node_head;
	} else {
		node_tail->next = new (std::nothrow) node(x1, y1, x2, y2, x3, y3);
		if(!node_tail->next) {
			cout << "Error, out of memory." << endl;
			exit(-1);
		}
		node_tail->next->prev = node_tail;
		node_tail = node_tail->next;
	}
	nodes++;
	
	add_edge(x1, y1, x2, y2, FOV);
	edge_tail->owner = (void *)node_tail;
	node_tail->set_fov_e(0, edge_tail);
	add_edge(x2, y2, x3, y3, FOV);
	edge_tail->owner = (void *)node_tail;
	node_tail->set_fov_e(1, edge_tail);
	add_edge(x3, y3, x1, y1, FOV);
	edge_tail->owner = (void *)node_tail;
	node_tail->set_fov_e(2, edge_tail);
}
		
void map::add_edge(double startx, double starty, double stopx, double stopy, edge_type etype) {
	if(!edge_head) {
		edge_head = new (std::nothrow) edge(startx, starty, stopx, stopy);
		if(!edge_head) {
			cout << "Error, out of memory." << endl;
			exit(-1);
		}
		edge_tail = edge_head;
	} else {
		edge_tail->next = new (std::nothrow) edge(startx, starty, stopx, stopy);
		if(!edge_tail->next) {
			cout << "Error, out of memory." << endl;
			exit(-1);
		}
		edge_tail->next->prev = edge_tail;
		edge_tail = edge_tail->next;
	}
	edge_tail->type = etype;
	edges++;
}

void map::bad_sort_max_x() {
	int i;
	for(i=1; i<edges; i++) {
		edge1 = edge_head;
		edge2 = edge1->next;
		while(edge2) {
			//check which edge is larger
			if(edge1->get_max_x() > edge2->get_max_x()) {
				//edge2 is smaller we must swap them
				if(edge1->prev) edge1->prev->next = edge2;
				if(edge2->next) edge2->next->prev = edge1;
				edge1->next = edge2->next;
				edge2->prev = edge1->prev;
				edge2->next = edge1;
				edge1->prev = edge2;
				
				//fix head or tail pointer if necessary
				if(edge1 == edge_head) {
					edge_head = edge2;
				}
				if(edge2 == edge_tail) {
					edge_tail = edge1;
				}
				
				//now move on to next edge
				edge2 = edge1->next;
			} else {
				//move on to next edge
				edge1 = edge2;
				edge2 = edge1->next;
			}
		}
	}
}

void map::bad_sort_max_y() {
	int i;
	for(i=1; i<edges; i++) {
		edge1 = edge_head;
		edge2 = edge1->next;
		while(edge2) {
			//check which edge is larger
			if(edge1->get_max_y() > edge2->get_max_y()) {
				//edge2 is smaller we must swap them
				if(edge1->prev) edge1->prev->next = edge2;
				if(edge2->next) edge2->next->prev = edge1;
				edge1->next = edge2->next;
				edge2->prev = edge1->prev;
				edge2->next = edge1;
				edge1->prev = edge2;
				
				//fix head or tail pointer if necessary
				if(edge1 == edge_head) {
					edge_head = edge2;
				}
				if(edge2 == edge_tail) {
					edge_tail = edge1;
				}
				
				//now move on to next edge
				edge2 = edge1->next;
			} else {
				//move on to next edge
				edge1 = edge2;
				edge2 = edge1->next;
			}
		}
	}
}
 
int map::check_sort_max_x() {
	edge1 = edge_head;
	edge2 = edge1->next;
	while(edge2) {
		if(edge1->get_max_x() > edge2->get_max_x()) {
			return 0;  //unsorted
		}
		edge1 = edge2;
		edge2 = edge1->next;
	}
	return 1; //sorted
}

int map::check_sort_min_x() {
	edge1 = edge_head;
	edge2 = edge1->next;
	while(edge2) {
		if(edge1->get_min_x() > edge2->get_min_x()) {
			return 0;  //unsorted
		}
		edge1 = edge2;
		edge2 = edge1->next;
	}
	return 1; //sorted
}

int map::check_stable() {
	int stability_checked = 0;
	edge1 = edge_head;
	edge2 = edge1->next;
	while(edge2) {
		if(edge1->get_max_x() > edge2->get_max_x()) {
			return 0;  //unsorted
		} else if(edge1->get_max_x() == edge2->get_max_x()) {
			stability_checked = 1;
			if(edge1->get_max_y() > edge2->get_max_y()) {
				return 1; //unstable
			}
		}
		edge1 = edge2;
		edge2 = edge1->next;
	}
	if(stability_checked) return 2; //stable
	return 3; //sorted, may or may not be stable
}
		
void map::sort_edges_max_x() {
	sort_edges(0);
}

void map::sort_edges_min_x() {
	sort_edges(2);
}

void map::sort_edges_max_y() {
	sort_edges(1);
}

void map::sort_edges_min_y() {
	sort_edges(3);
}

void map::sort_edges(int sort_type) {
	int i, e1size, e2size, nmerges, insize = 1, edge1greater = 0;

	while(1) {
		nmerges = 0;
		edge1 = edge_head;
		newhead = NULL;
		newtail = NULL;
		
		while(edge1) {
			nmerges++;	//counts number of merges occur at current insize
			edge2 = edge1;
			e1size = 0;
			for(i=0;i<insize;i++){
				e1size++;
				edge2 = edge2->next;
				if(!edge2) break;	//if the edge_tail is hit, edge2 would be null
			}

			e2size = insize;		//e1size is all that really matters, since nothing
							//would happen if edge2 == NULL

			while (e1size > 0 || (e2size > 0 && edge2)) {

				switch(sort_type) {
					//uses appropriate values to determine what value should be popped
					case 0: if(edge1 != NULL && edge2 != NULL) {
					// max_x
							if(edge1->get_max_x() == edge2->get_max_x())
							edge1greater = edge1->get_max_y() > edge2->get_max_y();
							else edge1greater = edge1->get_max_x() > edge2->get_max_x();
					  	}
						break;
					case 1: if(edge1 != NULL && edge2 != NULL) {
					// max_y
							if(edge1->get_max_y() == edge2->get_max_y())
								edge1greater = edge1->get_max_x() > edge2->get_max_x();
							else edge1greater = edge1->get_max_y() > edge2->get_max_y();
						}
						break;
					case 2: if(edge1 != NULL && edge2 != NULL) {
					// min_x
							if(edge1->get_min_x() == edge2->get_min_x())
							edge1greater = edge1->get_min_y() > edge2->get_min_y();
							else edge1greater = edge1->get_min_x() > edge2->get_min_x();
						}
						break;
					case 3: if(edge1 != NULL && edge2 != NULL) {
					// min_y
							if(edge1->get_min_y() == edge2->get_min_y())
							edge1greater = edge1->get_min_x() > edge2->get_min_x();
							else edge1greater = edge1->get_min_y() > edge2->get_min_y();
						}
						break;
					default:cout << "Invalid Sort Type.\n";
						exit(1);
						break;
				}

				if(e1size == 0) {
				//pops rest of edge2 (e1size was limited by edge2 reaching the tail)
					edge2 = pop_edge(edge2);
					e2size--;
				} else if(e2size == 0 || !edge2) {
				//pops rest of edge1
					edge1 = pop_edge(edge1);
					e1size--;
				} else if(edge1greater == 1) {
				//pops lesser value
					edge2 = pop_edge(edge2);
					e2size--;
				} else {
					edge1 = pop_edge(edge1);
					e1size--;
				}
			}

			edge1 = edge2;
		}

		//reassigns linked list head/tail to new structure
		edge_head = newhead;
		edge_tail = newtail;
		if(nmerges <= 1) return;

		//if nmerges <= 1, list should be sorted
		//otherwise, check lists twice the size
		insize *= 2;
	}
}

void map::print_map(int sort_type) {
	print_map(edge_head, sort_type);
}

void map::print_map(edge *tmpedge, int sort_type) {
	cout << "MAP BEGIN " << sort_type << "\n";

	while(tmpedge) {
		switch(sort_type) {
			case 0	:	cout << tmpedge->get_max_x();
					break;
			case 1	:	cout << tmpedge->get_max_y();
					break;
			case 2	:	cout << tmpedge->get_min_x();
					break;
			case 3	:	cout << tmpedge->get_min_y();
					break;
		}
		cout << endl;
		tmpedge = tmpedge->next;
	}

	cout << "MAP END " << sort_type << "\n";
}

edge* map::pop_edge(edge *edgex) {
	//edgex is smaller
	//pop edgex and throw at end of list
	//returns next edge
	if(edgex->prev) edgex->prev->next = edgex->next;
	else if(edgex == edge_head) {
		edge_head = edgex->next;
		if(edge_head) edge_head->prev = NULL;
	}
	if(edgex->next) edgex->next->prev = edgex->prev;
	if(!newhead) {
	//start new linked list
		newhead = edgex;
		edgex = edgex->next;
		newhead->prev = NULL;
		newhead->next = NULL;
		newtail = newhead;
	} else {
	//append edge to tail of new linked list
		newtail->next = edgex;
		edgex->prev = newtail;
		newtail = edgex;
		edgex = edgex->next;
		newtail->next = NULL;
	}

	return edgex;
}

void map::sort_nodes() {
	int i, e1size, e2size, nmerges, insize = 1, node1greater = 0;

	while(1) {
		nmerges = 0;
		node1 = node_head;
		new_node_head = NULL;
		new_node_tail = NULL;
		
		while(node1) {
			nmerges++;	//counts number of merges occur at current insize
			node2 = node1;
			e1size = 0;
			for(i=0;i<insize;i++){
				e1size++;
				node2 = node2->next;
				if(!node2) break;	//if the node_tail is hit, node2 would be null
			}

			e2size = insize;		//e1size is all that really matters, since nothing
							//would happen if node2 == NULL

			while (e1size > 0 || (e2size > 0 && node2)) {

				if(node1 != NULL && node2 != NULL) {
					node1greater = node1->get_min_x() > node2->get_min_x();
				}

				if(e1size == 0) {
				//pops rest of node2 (e1size was limited by node2 reaching the tail)
					node2 = pop_node(node2);
					e2size--;
				} else if(e2size == 0 || !node2) {
				//pops rest of node1
					node1 = pop_node(node1);
					e1size--;
				} else if(node1greater == 1) {
				//pops lesser value
					node2 = pop_node(node2);
					e2size--;
				} else {
					node1 = pop_node(node1);
					e1size--;
				}
			}

			node1 = node2;
		}

		//reassigns linked list head/tail to new structure
		node_head = new_node_head;
		node_tail = new_node_tail;
		if(nmerges <= 1) return;

		//if nmerges <= 1, list should be sorted
		//otherwise, check lists twice the size
		insize *= 2;
	}
}

node* map::pop_node(node *nodex) {
	//nodex is smaller
	//pop nodex and throw at end of list
	//returns next node
	if(nodex->prev) nodex->prev->next = nodex->next;
	else if(nodex == node_head) {
		node_head = nodex->next;
		if(node_head) node_head->prev = NULL;
	}
	if(nodex->next) nodex->next->prev = nodex->prev;
	if(!new_node_head) {
	//start new linked list
		new_node_head = nodex;
		nodex = nodex->next;
		new_node_head->prev = NULL;
		new_node_head->next = NULL;
		new_node_tail = new_node_head;
	} else {
	//append node to tail of new linked list
		new_node_tail->next = nodex;
		nodex->prev = new_node_tail;
		new_node_tail = nodex;
		nodex = nodex->next;
		new_node_tail->next = NULL;
	}

	return nodex;
}

//note it may be better to line sweep differently
//currently nodes are not removed from active
//if there are still nodes before them that
//may overlap, it is not clear which way
//would be faster
void map::add_overlap_edges() {
	if(grid == 0) {
		if(!only_fov) {
			//first sort nodes
			sort_nodes();
		
			//queue of active nodes
			node **active;
			active = NULL;
			active = new (std::nothrow)node*[nodes];
			if(!active) {
				cout << "Error, out of memory." << endl;
				exit(-1);
			}
			
			//head points to oldest, tail to newest
			int i, j, active_head, active_tail;
			double current_min;
			active_head = 0;
			active_tail = 0;
			node1 = node_head;
			for(i=0; i<nodes; i++) {
				//add current node to active
				active[i] = node1;
				current_min = node1->get_min_x();
				
				//remove any from the list that can
				//no longer overlap
				for(j=active_head; j<active_tail; j++) {
					if(active[j]->get_max_x() <= current_min) active_head++;
					else j = active_tail;
				}
				
				//sort through the active list, checking for overlaps
				for(j=active_head; j<active_tail; j++) {
					if(node1->overlaps(active[j])) {
						//do some overlap stuff
						add_semi_voronoi(node1, active[j]);
					}
				}
				//move to next
				node1 = node1->next;
				active_tail++;
			}
			delete [] active;
		}
	} else {
		double i,j;
		for(i=0; i<width; i=i+grid) {
			for(j=0; j<height; j=j+grid) {
			//make horizontal edge
				if(i+grid > width) {
					add_edge(i, j, width, j, VORONOI);
				} else {
					add_edge(i, j, i+grid, j, VORONOI);
				}
			
			//make vertical edge
				if(j+grid > height) {
					add_edge(i, j, i, height, VORONOI);
				} else {
					add_edge(i, j, i, j+grid, VORONOI);
				}
			}
			
			//make top horizontal edge
			if(i+grid > width) {
				add_edge(i, height, width, height, VORONOI);
			} else {
				add_edge(i, height, i+grid, height, VORONOI);
			}
		}
		
	//make end vertical edges
		for(j=0; j<height; j=j+grid) {
			if(j+grid > height) {
				add_edge(width, j, width, height, VORONOI);
			} else {
				add_edge(width, j, width, j+grid, VORONOI);
			}
		}
	}
}

//bad algorithm to intersect edges
/*
void map::intersect_edges() {
	point *intersect;
	edge1 = edge_head;
	while(edge1) {
		edge2 = edge1->next;
		while(edge2) {
			//check for intersection
			intersect = edge1->intersect_point(edge2);
			if(intersect) {
				//split edges
				if(!edge1->end_point(intersect)) {
					//split edge
					edge3 = new (std::nothrow) edge(intersect, edge1->get_stop());
					if(!edge3) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					edge1->set_stop(intersect);
					
					//add new edge to end of list
					edge_tail->next = edge3;
					edge3->prev = edge_tail;
					edge_tail = edge3;
					edges++;
				}
				if(!edge2->end_point(intersect)) {
					//split edge
					edge3 = new (std::nothrow) edge(intersect, edge2->get_stop());
					if(!edge3) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					edge2->set_stop(intersect);
					
					//add new edge to end of list
					edge_tail->next = edge3;
					edge3->prev = edge_tail;
					edge_tail = edge3;
					edges++;
				}
				
				intersect_x[intersect_count] = intersect->x;
				intersect_y[intersect_count] = intersect->y;
				intersect_count++;
				
			}
			edge2 = edge2->next;
		}
		edge1 = edge1->next;
	}
	cout << intersect_count++ << endl;
}
*/

//note it may be better to line sweep differently
//currently nodes are not removed from active
//if there are still nodes before them that
//may overlap, it is not clear which way
//would be faster
void map::intersect_edges() {
	//first sort edges
	sort_edges_min_x();

	//queue of active edges
	edge **active;
	active = NULL;
	active = new (std::nothrow)edge*[edges*edges]; //edges^2 is max through intersecting
	if(!active) {
		cout << "Error, out of memory." << endl;
		exit(-1);
	}
	
	//head points to oldest, tail to newest
	int i, j, active_head, active_tail;
	double current_min;
	active_head = 0;
	active_tail = 0;
	edge1 = edge_head;
	point *intersect;
	point *t_point;
	for(i=0; i<edges; i++) {
		//if(i==3) exit(0);
		//if(i>10) exit(-1);
		//cout << "i:" << i << endl;
		//cout << "edges:" << edges << endl;
		//add current node to active
		active[i] = edge1;
		current_min = edge1->get_min_x();
		
		//remove any from the list that can
		//no longer overlap
		for(j=active_head; j<active_tail; j++) {
			if((current_min - active[j]->get_max_x()) > DELTA) active_head++;
			//if(active[j]->get_max_x() <= current_min) active_head++;
			else j = active_tail;
		}
		
		//sort through the active list, checking for overlaps
		for(j=active_head; j<active_tail; j++) {
			//cout << "j:" << j << endl;
			//cout << "edges:" << edges << endl;
			intersect = edge1->intersect_point(active[j]);
			if(intersect != NULL) {
				//they do intersect, split both edges
				//cout << "intersect: " << intersect->x << "," << intersect->y << endl;
				
				//split edge then find correct place to insert it, in order to keep sorted
				//second edge may have to check to increase active tail or head
				
				//intersect_x[intersect_count] = intersect->x;
				//intersect_y[intersect_count] = intersect->y;
				//intersect_count++;
				
				//find spot to insert new edges
				edge3 = edge1;
				while(edge3->next && (edge3->next->get_min_x() < intersect->x)) edge3 = edge3->next;
				
				if(!edge1->end_point(intersect)) {
					//cout << "edge1: " << edge1->get_start_x() << "," << edge1->get_start_y()
					//		<< " " << edge1->get_stop_x() << "," << edge1->get_stop_y() << endl;
					//split edge
					t_point = new (std::nothrow) point;
					if(!t_point) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					t_point->x = intersect->x;
					t_point->y = intersect->y;
					edge2 = new (std::nothrow) edge(t_point, edge1->get_stop());
					if(!edge2) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					//edge1->set_stop(intersect);
					t_point = new (std::nothrow) point;
					if(!t_point) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					t_point->x = intersect->x;
					t_point->y = intersect->y;
					edge1->set_stop(t_point);
					edge2->type = edge1->type;
					edge2->owner = edge1->owner;
					
					//now new edge goes right after current edge
					if(edge3->next) edge3->next->prev = edge2;
					else edge_tail = edge2;
					edge2->next = edge3->next;
					edge2->prev = edge3;
					edge3->next = edge2;
					edges++;
				}
				
				if(!active[j]->end_point(intersect)) {
					//cout << "active: " << active[j]->get_start_x() << "," << active[j]->get_start_y()
					//		<< " " << active[j]->get_stop_x() << "," << active[j]->get_stop_y() << endl;
					//split edge
					t_point = new (std::nothrow) point;
					if(!t_point) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					t_point->x = intersect->x;
					t_point->y = intersect->y;
					edge2 = new (std::nothrow) edge(t_point, active[j]->get_stop());
					if(!edge2) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					//active[j]->set_stop(intersect);
					t_point = new (std::nothrow) point;
					if(!t_point) {
						cout << "Error, out of memory." << endl;
						exit(-1);
					}
					t_point->x = intersect->x;
					t_point->y = intersect->y;
					active[j]->set_stop(t_point);
					edge2->type = active[j]->type;
					edge2->owner = active[j]->owner;
					
					//now new edge goes right after current edge
					if(edge3->next) edge3->next->prev = edge2;
					else edge_tail = edge2;
					edge2->next = edge3->next;
					edge2->prev = edge3;
					edge3->next = edge2;
					edges++;
				}
				
				delete intersect;
				
				//if(active[j]->next) active[j]->next->prev = edge2;
				//edge2->next = active[j]->next;
				//edge2->prev = active[j];
				//active[j]->next = edge2;
				//edges++;
				
				//now add edge2 to active, we should add it to the correct place in
				//the active list, however this is hard to do, so we add it to the end
			}
		}
		//move to next
		edge1 = edge1->next;
		active_tail++;
	}
	//cout << intersect_count << endl;
	delete [] active;
	
	//now add unweighted start edges
	double STARTX = -100;
	double ENDX = width+100;
	edge1 = edge_head;
	while(edge1) {
		if(fabs(edge1->get_start_x()) < DELTA && fabs(edge1->get_stop_x()) < DELTA) {
			//add a start edge
			add_edge(STARTX, height/2, 0, edge1->get_start_y(), START);
			add_edge(STARTX, height/2, 0, edge1->get_stop_y(), START);
		}
		if(fabs(edge1->get_start_x()-width) < DELTA && fabs(edge1->get_stop_x()-width) < DELTA) {
			add_edge(width, edge1->get_start_y(), ENDX, height/2, START);
			add_edge(width, edge1->get_stop_y(), ENDX, height/2, START);
		}
		edge1 = edge1->next;
	}
}

void map::update_weights() {
	//first sort nodes and edges
	sort_edges_min_x();
	sort_nodes();

	//queue of active nodes
	node **active;
	active = NULL;
	active = new (std::nothrow)node*[nodes];
	if(!active) {
		cout << "Error, out of memory." << endl;
		exit(-1);
	}

	node *active_head, *active_tail;
	point p;
	active_head = node_head;
	active_tail = node_head;
	edge1 = edge_head;
	while(edge1) {
		if(edge1->type != START) {
		//find midpoint
		p.x = (edge1->get_start_x() + edge1->get_stop_x()) / 2;
		p.y = (edge1->get_start_y() + edge1->get_stop_y()) / 2;
		
		//add nodes to active list
		while(active_tail && active_tail->get_min_x() < edge1->get_max_x()) active_tail = active_tail->next;
		
		//remove nodes from the active list
		while(active_head && active_head->get_max_x() < edge1->get_min_x()) active_head = active_head->next;
		
		//cycle through active list of nodes
		node1 = active_head;
		while(node1 != active_tail) {
			//check if edge is in fov
			if((edge1->owner != (void *)node1) && node1->inside(&p)) {
				//update weight
				edge1->update_weight(node1->get_fov(0));
			}
			
			node1 = node1->next;
		}
		} else {
			edge1->weight = INF;
		}
		edge1 = edge1->next;
	}
	
	delete [] active;
}


void map::add_semi_voronoi(node *n1, node *n2) {
	//we add the 'voronoi' edge between the two nodes in the overlapping region
	//we do not split any edges yet
	
	//perp bisector has a point at (x1+x2)/2, (y1+y2)/2
	//and if slope of segment is m, then slope of pb is -1/m
	
	//perp bisector
	//calculate slope and 1 point on line
	double m;
	if(n2->get_y() == n1->get_y()) {
		//straight up and down
		m = 100000;
	}
	else m = (n1->get_x()-n2->get_x())/(n2->get_y()-n1->get_y());
	double x3 = (n1->get_x()+n2->get_x())/2;
	double y3 = (n1->get_y()+n2->get_y())/2;
	
	//now find 6 intersections with 2 fovs
	point *int1[3];
	point *int2[3];
	int i;
	for(i=0; i<3; i++) {
		int1[i] = n1->intersect(m,x3,y3,i);
		int2[i] = n2->intersect(m,x3,y3,i);
	}
	
	//find which are inside the other
	//int inside[6];
	double x1, x2, y1, y2;
	x1=0;
	x2=0;
	y1=0;
	y2=0;
	//int edge = 0;
	int count = 0;
	for(i=0; i<3; i++) {
		if(int1[i] != NULL) {
			//intersect_x[intersect_count] = int1[i]->x;
			//intersect_y[intersect_count] = int1[i]->y;
			//intersect_count++;
			
			//inside[i] = n2->inside(int1[i]);
			if(n2->inside(int1[i])) {
				if(count == 0) {
					x1 = int1[i]->x;
					y1 = int1[i]->y;
					count++;
				} else {
					//check to make sure it is not a repeat
					if(fabs(x1-int1[i]->x) > DELTA || fabs(y1-int1[i]->y) > DELTA) {
						//edge = 1;
						x2 = int1[i]->x;
						y2 = int1[i]->y;
						count++;
					}
				}
			}
		}
		if(int2[i] != NULL) {
			//intersect_x[intersect_count] = int2[i]->x;
			//intersect_y[intersect_count] = int2[i]->y;
			//intersect_count++;
			
			//inside[i+3] = n1->inside(int2[i]);
			if(n1->inside(int2[i])) {
				if(count == 0) {
					x1 = int2[i]->x;
					y1 = int2[i]->y;
					count++;
				} else {
					//check to make sure it is not a repeat
					if(fabs(x1-int2[i]->x) > DELTA || fabs(y1-int2[i]->y) > DELTA) {
						//edge = 1;
						x2 = int2[i]->x;
						y2 = int2[i]->y;
						count++;
					}
				}
			}
		}
	}
	
	//note count==1 is ok this is a line that is shorter than delta
	if(count != 0 && count != 1 && count != 2) {
		cout << "Oops voronoi intersection count=" << count << endl;
		//print out node info
		cout << "node 1: " << n1->get_x() << "," << n1->get_y() << endl;
		cout << "node 2: " << n2->get_x() << "," << n2->get_y() << endl;
	}
	
	if(count == 2) add_edge(x1, y1, x2, y2, VORONOI);
	
	for(i=0; i<3; i++) {
		if(int1[i]) delete int1[i];
		if(int2[i]) delete int2[i];
	}
}

void map::add_border() {
	add_edge(0, 0, 0, height, EXTRA);
	add_edge(0, height, width, height, EXTRA);
	add_edge(width, height, width, 0, EXTRA);
	add_edge(width, 0, 0, 0, EXTRA);
}

//for now use seapahn's graph search
double map::find_maximal_breach() {
	combine_vertices();
	
	//add start and stop points and edges
	if(!GV) {
		GV = new (std::nothrow)Graph;
		if(!GV) {
			cout << "Error, Out of memory\n";
			exit(-1);
		}
	}
	GV->MAX_VERTICES = vertices+1;
	GV->initialize();
	//GV->reset();
	
	edge1 = edge_head;
	double minweight = INF;
	double maxweight = 0;
	while(edge1) {
		if(edge1->weight > maxweight) maxweight = edge1->weight;
		if(edge1->weight < minweight) minweight = edge1->weight;
		GV->AddEdge((int)edge1->get_start(),(int)edge1->get_stop(),edge1->weight);
		edge1 = edge1->next;
	}
	
	//find start and stop points of graph
	point *start = NULL;
	point *stop = NULL;
	double STARTX = -100;
	double ENDX = width+100;
	vertice1 = vertice_head;
	while(vertice1 != NULL) {
		//cout << "1\n";
		if(fabs(vertice1->y-(height/2.0)) <= DELTA) {
		//if(fabs(vertice1->y-(0)) <= DELTA) {	
			if(fabs(vertice1->x-STARTX) <= DELTA) {
				//found start
				start = vertice1;
				if(stop != NULL) vertice1 = NULL;
				else vertice1 = vertice1->next;
			} else if(fabs(vertice1->x-(ENDX)) <= DELTA) {
				//found end
				stop = vertice1;
				if(start != NULL) vertice1 = NULL;
				else vertice1 = vertice1->next;
			} else vertice1 = vertice1->next;
		} else vertice1 = vertice1->next; 
	}

	if(start==NULL) {
		cout << "Error, unable to find start\n";
		exit(-1);
	}
	if(stop==NULL) {
		cout << "Error, unable to find stop\n";
		exit(-1);
	}

	worst_breach = GV->MaxPath((int)start,(int)stop,minweight,maxweight);
	
	return worst_breach;
}

void map::combine_vertices() {
	int i;
	int start_match;
	int stop_match;
	edge1 = edge_head;
	point *t_point;
	vertices = 0;
	while(edge1) {
		if(vertices == 0) {
			vertice_head = edge1->get_start();
			vertice_tail = edge1->get_stop();
			vertice_head->next = vertice_tail;
			//vertice_head = new (std::nothrow) point_list(edge1->get_start());
			//if(!vertice_head) {
			//	cout << "Error, out of memory" << endl;
			//	exit(-1);
			//}
			//vertice_head->add_to_end(edge1->get_stop());
			vertices = 2;
		} else {
			start_match = 0;
			stop_match = 0;
			vertice1 = vertice_head;
			for(i=0; i<vertices; i++) {
				if((fabs(edge1->get_start_x()-vertice1->x) < DELTA) &&
								(fabs(edge1->get_start_y()-vertice1->y) < DELTA)) {
					//matching vertice
					start_match = 1;
					t_point = edge1->get_start();
					delete t_point;
					edge1->set_start(vertice1);
				}
				if((fabs(edge1->get_stop_x()-vertice1->x) < DELTA) &&
								(fabs(edge1->get_stop_y()-vertice1->y) < DELTA)) {
					//matching vertice
					stop_match = 1;
					t_point = edge1->get_stop();
					delete t_point;
					edge1->set_stop(vertice1);
				}
				//vertice1 = vertice1->get_next();
				vertice1 = vertice1->next;
			}
			if(!start_match) {
				//no match add new vertice
				vertices++;
				//vertice_head->add_to_end(edge1->get_start());
				vertice_tail->next = edge1->get_start();
				vertice_tail = vertice_tail->next;
			}
			if(!stop_match) {
				//no match add new vertice
				vertices++;
				vertice_tail->next = edge1->get_stop();
				vertice_tail = vertice_tail->next;
				//vertice_head->add_to_end(edge1->get_stop());
			}
		}
		edge1 = edge1->next;
	}
}

//void map::construct_graph() {
//	
//}

void map::move(double dt) {
	int i;
	//move all nodes
	for(i=0; i<nodes; i++) {
		simple_map[i].move(dt, width, height, movement_on, rotation_on);
	}
	
	//recalculate breach
	refresh_structures();
}

void map::refresh_structures() {
	int i;
	//delete old structures

	//delete edges
	edges = 0;
	edge_tail = NULL;
	if(edge_head) edge_head->delete_all();
	edge_head = NULL;
	
	//cout << "b\n";
	//delete points
	vertices = 0;
	vertice_tail = NULL;
	if(vertice_head) vertice_head->delete_all();
	vertice_head = NULL;
	
	//cout << "c\n";
	//delete nodes
	double temp = nodes;
	nodes = 0;
	node_tail = NULL;
	if(node_head) node_head->delete_all();
	node_head = NULL;
	
	//cout << "d\n";
	//add new structures
	for(i=0; i<temp; i++) {
		//x,y,start angle, sweep angle, range
		add_node(simple_map[i].x, simple_map[i].y, simple_map[i].initial_angle,
						 simple_map[i].sweep_angle, simple_map[i].range);
	}
	
	add_border();
	//recalculate
	add_overlap_edges();
	intersect_edges();
	remove_outside_edges();
	update_weights();
	find_maximal_breach();
}

#ifdef GUI
void map::draw() {
	
	edge1 = edge_head;
	while(edge1 != NULL) {
		switch(edge1->type) {
			case FOV:
				if(fov_on) {
					glLineWidth(fov_size);
					//glColor3f(0,0,1);
					//cout << edge1->weight << endl;
					if(edge1->weight == INF || !weight_color_on) glColor3f(0,0,1);
					else glColor3f(1-edge1->weight/max_range, edge1->weight/max_range, 0);
					glBegin(GL_LINES);
					glVertex2f(edge1->get_start_x(), edge1->get_start_y());
					glVertex2f(edge1->get_stop_x(), edge1->get_stop_y());
					glEnd();
				}
				break;
			case VORONOI:
				if(voronoi_on) {
				glLineWidth(voronoi_size);
				//glColor3f(1,0,0);
				//cout << edge1->weight << endl;
				if(edge1->weight == INF || !weight_color_on) glColor3f(1,0,1);
				else glColor3f(1-edge1->weight/max_range, edge1->weight/max_range, 0);
				glBegin(GL_LINES);
				glVertex2f(edge1->get_start_x(), edge1->get_start_y());
				glVertex2f(edge1->get_stop_x(), edge1->get_stop_y());
				glEnd();
				}
				break;
			case PATH_FOV:
				break;
			case PATH_VORONOI:
				break;
			case START:
				glLineWidth(1);
				//glColor3f(1,0,0);
				//cout << edge1->weight << endl;
				if(edge1->weight == INF || !weight_color_on) glColor3f(1,1,1);
				else glColor3f(1-edge1->weight/max_range, edge1->weight/max_range, 0);
				glBegin(GL_LINES);
				glVertex2f(edge1->get_start_x(), edge1->get_start_y());
				glVertex2f(edge1->get_stop_x(), edge1->get_stop_y());
				glEnd();
				break;
			case EXTRA:
				glLineWidth(1);
				//glColor3f(1,0,0);
				//cout << edge1->weight << endl;
				if(edge1->weight == INF || !weight_color_on) glColor3f(1,1,1);
				else glColor3f(1-edge1->weight/max_range, edge1->weight/max_range, 0);
				glBegin(GL_LINES);
				glVertex2f(edge1->get_start_x(), edge1->get_start_y());
				glVertex2f(edge1->get_stop_x(), edge1->get_stop_y());
				glEnd();
				break;
		};
		edge1 = edge1->next;
	}
	
	if(nodes_on) {
		node1 = node_head;
		glColor3f(0.0, 1.0, 0.0);
		glPointSize(node_size);
		glBegin(GL_POINTS);
		while(node1 != NULL) {
			glVertex2f(node1->get_x(), node1->get_y());
			node1 = node1->next;
		}
		glEnd();
	}
	
	int i;
	glColor3f(1.0, 1.0, 1.0);
	glPointSize(4.0);
	glBegin(GL_POINTS);
	for(i=0; i<intersect_count; i++) {
		glVertex2f(intersect_x[i], intersect_y[i]);
	}
	glEnd();
	
	if(path_on) {
		glLineWidth(path_size);
	
		glBegin(GL_LINE_STRIP);
		for (i=0; i<GV->p_count; ++i) {
			//if(!bw) {
				if ((i==GV->critical || i==GV->critical-1) && worst_breach < INF) {
					glColor3f(1.0f,1.0f,0.0f);
				}
				else
					glColor3f(0.0f,0.7f,0.7f);
			//} else glColor3f(path_color, path_color, path_color);
			//find matching vertex
			vertice1 = vertice_head;
			while(vertice1 != NULL) {
				if((int)vertice1 == GV->P[i]) {
					glVertex2f(vertice1->x, vertice1->y);
					vertice1 = NULL;
				} else vertice1 = vertice1->next;
			}
		}
		glEnd();
		glLineWidth(1);
	}
	
	//display breach
#ifdef LINUX_COMPILE
	char str[32];
	glColor3f(1.0,1.0,1.0);
	glRasterPos2f(-95, 0);
	if(worst_breach == INF) {
		glutBitmapCharacter(GLUT_BITMAP_9_BY_15,'F');
		glutBitmapCharacter(GLUT_BITMAP_9_BY_15,'U');
		glutBitmapCharacter(GLUT_BITMAP_9_BY_15,'L');
		glutBitmapCharacter(GLUT_BITMAP_9_BY_15,'L');
	} else {
		sprintf(str, "%f", worst_breach);
		i = 0;
		while(str[i] != 0 && i<6) {
			glutBitmapCharacter(GLUT_BITMAP_9_BY_15,str[i]);
			i++;
		}
	}
#endif
		
		//display edge endpoints
		/*
		edge1 = edge_head;
		glColor3f(1.0, 1.0, 1.0);
		glPointSize(3.0);
		glBegin(GL_POINTS);
		while(edge1 != NULL) {
			if(edge1->type == EXTRA) {
				glVertex2f(edge1->get_start_x(), edge1->get_start_y());
				glVertex2f(edge1->get_stop_x(), edge1->get_stop_y());
			}
			edge1 = edge1->next;
		}
		glEnd();
		*/
	//}

	/*
	glColor3f(1.0, 1.0, 1.0);
	glBegin(GL_POINTS);
	glVertex2f(185.0,395.0);
	glVertex2f(150.27,591.962);
	glVertex2f(-2.93852,253.404);
	glVertex2f(113.0,450.0);
	glVertex2f(286.205,350.0);
	glVertex2f(286.205,213.0);
	glEnd();
	*/
	/*
#ifdef GUI  
	if(voronoi_on) {
	glLineWidth(voronoi_size);
	current_edge = edges;
	while(current_edge != NULL) {
		if(!bw) {
			if(current_edge->weight != MAX_WEIGHT) {
				glColor3f(current_edge->weight/range[0], current_edge->weight/range[0], 0);
			} else {
				glColor3f(1, 0, 0);
			}
		} else glColor3f(voronoi_color, 0.0, 0.0);
		glBegin(GL_LINES);
		glVertex2f(current_edge->start_x, current_edge->start_y);
		glVertex2f(current_edge->stop_x, current_edge->stop_y);
		glEnd();
		current_edge = current_edge->next;
	}
	}
  
	if(fov_on) {
		glLineWidth(fov_size);
		current_fov = fovs;
		while(current_fov != NULL) {
			if(!bw) {
				if(current_fov->weight != MAX_WEIGHT) {
					glColor3f(current_fov->weight/range[0], current_fov->weight/range[0], 0);
				} else {
					glColor3f(0, 1, 0);
				}
			} else glColor3f(0.0, 0.0, fov_color);
			glBegin(GL_LINES);
			glVertex2f(current_fov->start_x, current_fov->start_y);
			glVertex2f(current_fov->stop_x, current_fov->stop_y);
			glEnd();
			current_fov = current_fov->next;
		}
	}
  
	if(points_on) {
		if(!bw) glColor3f(0,0,1);
		else glColor3f(0.0, point_color, 0.0);
		glPointSize(point_size);
		glBegin(GL_POINTS);
		for(i=0; i<nodes; i++) {
			glVertex2i(x[i],y[i]);
		}
		glEnd();
	}
  
	if(int_on) {
		current_point = points;
		if(!bw) glColor3f(1,1,1);
		else glColor3f(intersect_color, intersect_color, intersect_color);
		glPointSize(intersect_size);
		glBegin(GL_POINTS);
		while(current_point != NULL) {
			glVertex2f(current_point->x, current_point->y);
			current_point = current_point->next;
		}
		glEnd();
	}
	
	if(path_on) {
		glLineWidth(path_size);
		glBegin(GL_LINE_STRIP);
		for (i=0; i<GV->p_count; ++i) {
			if(!bw) {
				if ((i==GV->critical || i==GV->critical-1) && worst_breach < MAX_WEIGHT) {
					glColor3f(1.0f,1.0f,0.0f);
				}
				else
					glColor3f(0.0f,0.7f,0.7f);
			} else glColor3f(path_color, path_color, path_color);
			//find matching vertex
			current_vert = verts;
			while(current_vert != NULL) {
				if((int)current_vert == GV->P[i]) {
					glVertex2f(current_vert->x, current_vert->y);
					current_vert = NULL;
				} else current_vert = current_vert->next;
			}
		}
		glEnd();
		glLineWidth(1);
	}
#endif
	*/
}
#endif
