// Exposure.cpp: implementation of the Exposure class.
//
// Seapahn Megerian (Meguerdichian)
// University of California at Los Angeles
// seapahn@cs.ucla.edu
//
// Here is where most of the exposure stuff happens.
// This class relies on the DJ_Graph class for the shortest path algo
//
//
// 1) Added run_side_to_side() that finds the min exposure path from one
//    side (x=0) of the field to the opposite side (x=MAX)
// 2) Modified run_current_grid(...) to take parameters for start and end
//    positions
// 3) Modified intensity function and added a new case to use the sensor
//    alpha and n parameters.
//////////////////////////////////////////////////////////////////////
#include "Exposure.h"

Exposure::Exposure() {
	NUM_SENSORS=0;
	g_res     = 1;
	g_div     = 1;
	sensor_model=SENSOR_MODEL_D2;
	intensity_model = INTENSITY_MODEL_ALL;
}

Exposure::~Exposure() {
}


void Exposure::init(unsigned ns, unsigned preset, unsigned s_model, unsigned i_model) {
	unsigned i, sdiv;
	sensor_model=s_model;
	intensity_model=i_model;

	// Determine if the new layout is destructive (new layout type)
	if (preset!=LAYOUT_KEEP_OLD && preset!=LAYOUT_INT_ERROR) NUM_SENSORS=ns;
	if (NUM_SENSORS==0) { NUM_SENSORS=30; preset=LAYOUT_RANDOM; }

	switch (preset) {
	case LAYOUT_PLUS:
		if (NUM_SENSORS%2==0) NUM_SENSORS--;
		sdiv=NUM_SENSORS/2;
		for (i=0; i<sdiv; i++) {
			sensor[i].x=(i*WIDTH)/(sdiv-1);
			sensor[i].y=HIGHT/2;
		}
		for (i=sdiv; i<NUM_SENSORS; i++){
			sensor[i].x=WIDTH/2;
			sensor[i].y=((i-sdiv)*HIGHT)/(sdiv-1);
		}
		break;
		

	case LAYOUT_SQUARE:
		sdiv=(unsigned) sqrt((double)NUM_SENSORS);
		if (sdiv<2) sdiv=2;
		NUM_SENSORS=sdiv*sdiv;

		for (i=1; i<NUM_SENSORS-1; i++) {
			sensor[i-1].x = ((i%sdiv)*WIDTH)/(sdiv-1);
			sensor[i-1].y = ((i/sdiv)*HIGHT)/(sdiv-1);
		}
		NUM_SENSORS-=2;
		break;

	case LAYOUT_TRIANGLE:
		sdiv=(unsigned) sqrt((double)NUM_SENSORS);
		if (sdiv==0) sdiv=1;
		NUM_SENSORS=sdiv*sdiv;

		for (i=0; i<NUM_SENSORS; i++) {
			sensor[i].x = ((i%sdiv)*WIDTH)/sdiv + ((i/sdiv)%2)*WIDTH/(2*sdiv);
			sensor[i].y = ((i/sdiv)*HIGHT)/sdiv + HIGHT/(2*sdiv);
		}
		for (i=0; i<sdiv/2; i++) {
			sensor[NUM_SENSORS+i].x = WIDTH;
			sensor[NUM_SENSORS+i].y = (2*i*HIGHT)/sdiv + HIGHT/(2*sdiv);
		}
		NUM_SENSORS+=sdiv/2;
		break;

	case LAYOUT_HEXAGON:
		sdiv=(unsigned) sqrt((double)NUM_SENSORS/3.0);
		if (sdiv==0) sdiv=1;
		NUM_SENSORS=3*sdiv*sdiv-1;

		for (i=0; i<NUM_SENSORS; i++) {
			// Don't even try to decypher this thing!
			// Basically, layout is the same as a grid, except pairs of points are pushed closer, alternating the order
			// on alternating lines.  Has the effect of creating hexagons.  The next line applies a correction factor.
			sensor[i].x = ((i%sdiv)*WIDTH)/sdiv + (2*(((i+((sdiv+1)%2)*(i/sdiv)+1)%2))-1)*(WIDTH/(4*sdiv)) + WIDTH/(4*sdiv);
			sensor[i].x = (int)((double)(WIDTH*sensor[i].x)/(double)(WIDTH-WIDTH/(2*sdiv)));
			sensor[i].y = (int)((double)(i/sdiv*HIGHT)/(double)(3*sdiv-1));
		}
		break;

	case LAYOUT_KEEP_OLD:
		break;

	case LAYOUT_INT_ERROR:
		for (i=0; i<NUM_SENSORS; i++) {
			// Add a uniform random error to original sensor position
			sensor[i].x =(int)( (double)sensor[i].real_x* (1-((2*(rand()%2)-1)*0.05)) );
			sensor[i].y =(int)( (double)sensor[i].real_y* (1-((2*(rand()%2)-1)*0.05)) );
			if (sensor[i].x<0     ) sensor[i].x=0;
			if (sensor[i].x>=WIDTH) sensor[i].x=WIDTH-1;
			if (sensor[i].y<0     ) sensor[i].y=0;
			if (sensor[i].y>=HIGHT) sensor[i].y=HIGHT-1;
		}
		break;


	default:		// LAYOUT_RANDOM ...
		for (i=0; i<NUM_SENSORS; i++) {
			sensor[i].x = rand()%WIDTH;
			sensor[i].y = rand()%HIGHT;
		}
		break;
	}

	data_valid=0;

	if (preset!=LAYOUT_INT_ERROR) {
		for (i=0; i<NUM_SENSORS; i++) {
			sensor[i].real_x=sensor[i].x;
			sensor[i].real_y=sensor[i].y;
		}
	}
}

void Exposure::set_grid(unsigned int grid_count, unsigned int grid_res) {
	g_res=grid_count;
	g_div=grid_res;
	if (g_res==0) g_res=1;
	if (g_div==0) g_div=1;
}


// Main exposure algo.  Create the search graph based on grid parameters and run
// the shortest path algorithm.  To save space, Each point on the grid is given a 
// unique name based on its position.  This name can be used to determine x and y
// positions of the point.
void Exposure::run_current_grid(unsigned start_row,unsigned start_col,unsigned dest_row,unsigned dest_col) {
	unsigned r,c,p,q,p_name,q_name;
	unsigned grid_point_count = (g_div+1)*(g_div+1);

	G.init();
	for (r=0; r<g_res; r++) {            			// grid row 
		for (c=0; c<g_res; c++) {		            // grid collumn
			for (p=0; p<grid_point_count; ) {       // points along gird square boundary
				p_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);

				if (p<=g_div || p>=(g_div+1)*g_div || (p%(g_div+1))==g_div) p++;
				else p+=g_div; // ((p%(g_div+1))==0    

				for (q=p; q<grid_point_count; )  {
					q_name = (r*g_div+q/(g_div+1))*(g_div*g_res+1) + c*g_div + q%(g_div+1);
					G.add_edge(p_name,q_name,grid_line_exposure(p_name,q_name));
					if (q<=g_div || q>=(g_div+1)*g_div || (q%(g_div+1))==g_div) q++;
					else q+=g_div; // ((q%(g_div+1))==0    
				}

			}
		}
	}
	
	if (start_row>=g_res) start_row=g_res-1;
	if (start_col>=g_res) start_col=g_res-1;
	if (dest_row >=g_res) dest_row =g_res-1;
	if (dest_col >=g_res) dest_col =g_res-1;

	// Starting point
	r=start_row;
	c=start_col;
	p=0;
	start_point_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);

	r=dest_row;
	c=dest_col;
	end_point_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);

	if (!G.do_Djikstra(start_point_name,end_point_name))
		data_valid=0;
	else
		data_valid=1;
}


// Same as run_current_grid with except this one cosideres all paths from one side to the other. 
// Adds dummy edges to the graph from a dummy source and sink and then removes them from the path.
void Exposure::run_side_to_side() {
	unsigned r,c,p,q,p_name,q_name;
	unsigned grid_point_count = (g_div+1)*(g_div+1);
	DJ_Graph_Node tn;

	G.init();
	for (r=0; r<g_res; r++) {            			// grid row 
		for (c=0; c<g_res; c++) {		            // grid collumn
			for (p=0; p<grid_point_count; ) {       // points along gird square boundary
				p_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);

				if (p<=g_div || p>=(g_div+1)*g_div || (p%(g_div+1))==g_div) p++;
				else p+=g_div; // ((p%(g_div+1))==0    

				for (q=p; q<grid_point_count; )  {
					q_name = (r*g_div+q/(g_div+1))*(g_div*g_res+1) + c*g_div + q%(g_div+1);
					G.add_edge(p_name,q_name,grid_line_exposure(p_name,q_name));
					if (q<=g_div || q>=(g_div+1)*g_div || (q%(g_div+1))==g_div) q++;
					else q+=g_div; // ((q%(g_div+1))==0    
				}

			}
		}
	}

	start_point_name = (g_res*g_div+1)*(g_res*g_div+1);   // next unused name
	end_point_name   = (g_res*g_div+1)*(g_res*g_div+1)+1; // next unused name

	// Add dummy edges
	for (r=0; r<g_res; r++) {            			// grid row 
		c=0;		// left side
		for (p=0; p<grid_point_count; p+=g_div+1) {       // points along gird square boundary
				p_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);
				G.add_edge(p_name,start_point_name,0);
		}

		c=g_res-1;
		for (p=g_div; p<grid_point_count; p+=g_div+1) {       // points along gird square boundary
				p_name = (r*g_div+p/(g_div+1))*(g_div*g_res+1) + c*g_div + p%(g_div+1);
				G.add_edge(p_name,end_point_name,0);
		}
	}
	
	if (!G.do_Djikstra(start_point_name,end_point_name))
		data_valid=0;
	else {											// now we fix the path (remove the dummy start and end)
		data_valid=1;
		tn.name=end_point_name;						// this is the node name at end of path (bottom right)
		if (G.nodes.Find(tn)) {
			DJ_Graph_Node *t=G.nodes.GetData();
 			if (t->prev) end_point_name=t->prev->name;
			while (t && t->prev && t->prev->prev!=t->prev) t=t->prev;
			t->prev=t;
		}
	}
}


// This function calculates the exposure integral over a light segment from s to d
// using E=length * average_exposure (of end points).
double Exposure::grid_line_exposure(unsigned s, unsigned d) {
	// These equations determine x and y coordinates from grid point names s and d.
	int    x1   = ((s%(g_div*g_res+1))*WIDTH)/(g_div*g_res);
	int    x2   = ((d%(g_div*g_res+1))*WIDTH)/(g_div*g_res);
	int    y1   = ((s/(g_div*g_res+1))*HIGHT)/(g_div*g_res);
	int    y2   = ((d/(g_div*g_res+1))*HIGHT)/(g_div*g_res);
	double int1 = intensity(x1,y1);       // Intensity (s)
	double int2 = intensity(x2,y2);       // Intensity (d)
	double dist = sqrt((double)((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1)));
	return ((dist*(int1+int2))/2.0);     
}

// Implements the sensor model (1/d, 1/d^2 ... )
double Exposure::sensor_strength(Sensor s, int x, int y) {
	double I,dp=1;
	double d=sqrt((double)((s.x-x)*(s.x-x)+(s.y-y)*(s.y-y)));
	if (d<0.000001) d=0.000001;

	switch(sensor_model) {
	case SENSOR_MODEL_D1 : I=1.0/d;          break;
	case SENSOR_MODEL_D2 : I=1.0/(d*d);      break;	
	case SENSOR_MODEL_D3 : I=1.0/(d*d*d);    break;
	case SENSOR_MODEL_D4 : I=1.0/(d*d*d*d);  break;
	default:			                            // case SENSOR_MODEL_PAR (use a and n)
		for (unsigned i=0; i<s.n; i++) dp *= d;          // find d^n
		if (dp<0.000001) dp=0.000001;
		I = s.a / dp;
		break;
	}
	return I;
}


// Implements the intensity model
double Exposure::intensity(int x, int y) {
	double I=0, d, min_d=1000000;
	unsigned i, close_s=0;

	switch (intensity_model) {
	case INTENSITY_MODEL_ALL:
		for (i=0; i<NUM_SENSORS; i++)
			I += sensor_strength(sensor[i],x,y);
		break;

	case INTENSITY_MODEL_CLOSEST:
		for (i=0; i<NUM_SENSORS; i++) {
			d=sqrt((double)((sensor[i].x-x)*(sensor[i].x-x)+(sensor[i].y-y)*(sensor[i].y-y)));
			if (d<min_d) { min_d=d; close_s=i; }
		}
		I = sensor_strength(sensor[close_s],x,y);
		break;
	}
	return I;
}



// Demonstrates how to use the results
void Exposure::Output() {
	int curr, x,y;

	if (!data_valid) return;

	double plen=0;
	int oldx=-1,oldy=-1;

	DJ_Graph_Node tn;
	tn.name=end_point_name;						// this is the node name at end of path (bottom right)
	
	if (G.nodes.Find(tn)) {
		DJ_Graph_Node *t=G.nodes.GetData();

		cout << "Exposure = " << t->weight << endl << "START ";
		while (t && t->prev!=t) {
			curr=t->name;
			x=(curr%(g_div*g_res+1))*WIDTH/(g_div*g_res);  // convert node name to x y coordinates
			y=(curr/(g_div*g_res+1))*HIGHT/(g_div*g_res);
			cout << "(" << x << "," << y << ")-";
			t=t->prev;

			// Keep track of length of path
			if (oldx>0)	plen += sqrt((double)((x-oldx)*(x-oldx)+(y-oldy)*(y-oldy)));
			oldx=x;
			oldy=y;
		}
		cout << "(" << x << "," << y << ") END" << endl;
		plen += sqrt((double)((oldx)*(oldx)+(oldy)*(oldy)));
	}
	cout << "Path Length= " << plen << endl;
}


